<?php

namespace AmgGroup\Tests;

use AmgGroup\ApiClient;
use GuzzleHttp\Psr7\Response;

/**
 * Test case for the ApiClient class
 */
class ApiClientTest extends TestCase
{
    /**
     * Test that getInstance returns the same instance for the same endpoint
     */
    public function testGetInstanceReturnsSameInstanceForSameEndpoint(): void
    {
        // Mock the authentication response
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));

        // Set up static properties
        ApiClient::setAuthURL('https://example.com/oauth/token');
        ApiClient::setClientID('test-client-id');
        ApiClient::setClientSecret('test-client-secret');
        ApiClient::setGrantType('client_credentials');
        ApiClient::setScope('read write');

        // Get two instances with the same endpoint
        $instance1 = ApiClient::getInstance('users');
        $instance2 = ApiClient::getInstance('users');

        // Assert they are the same instance
        $this->assertSame($instance1, $instance2);
    }

    /**
     * Test that getInstance returns different instances for different endpoints
     */
    public function testGetInstanceReturnsDifferentInstancesForDifferentEndpoints(): void
    {
        // Mock the authentication responses
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));

        // Set up static properties
        ApiClient::setAuthURL('https://example.com/oauth/token');
        ApiClient::setClientID('test-client-id');
        ApiClient::setClientSecret('test-client-secret');
        ApiClient::setGrantType('client_credentials');
        ApiClient::setScope('read write');

        // Get instances with different endpoints
        $instance1 = ApiClient::getInstance('users');
        $instance2 = ApiClient::getInstance('posts');

        // Assert they are different instances
        $this->assertNotSame($instance1, $instance2);
    }

    /**
     * Test the dry run mode functionality
     */
    public function testDryRunMode(): void
    {
        // Mock the authentication response
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));

        // Set up static properties
        ApiClient::setAuthURL('https://example.com/oauth/token');
        ApiClient::setClientID('test-client-id');
        ApiClient::setClientSecret('test-client-secret');
        ApiClient::setGrantType('client_credentials');
        ApiClient::setScope('read write');

        // Get an instance
        $apiClient = ApiClient::getInstance('users');

        // Test default dry run mode (should be true)
        $this->assertTrue($apiClient->isDry());

        // Test setting dry run mode to false
        $apiClient->setDry(false);
        $this->assertFalse($apiClient->isDry());

        // Test setting dry run mode back to true
        $apiClient->setDry(true);
        $this->assertTrue($apiClient->isDry());
    }

    /**
     * Test the list method
     */
    public function testList(): void
    {
        // Mock the authentication response
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));

        // Mock the list response
        $mockData = [
            'value' => [
                ['id' => 1, 'name' => 'User 1'],
                ['id' => 2, 'name' => 'User 2'],
                ['id' => 3, 'name' => 'User 3']
            ]
        ];
        $this->addMockResponse(200, [], json_encode($mockData));

        // Set up static properties
        ApiClient::setAuthURL('https://example.com/oauth/token');
        ApiClient::setClientID('test-client-id');
        ApiClient::setClientSecret('test-client-secret');
        ApiClient::setGrantType('client_credentials');
        ApiClient::setScope('read write');

        // Get an instance
        $apiClient = ApiClient::getInstance('users');

        // Call the list method
        $result = $apiClient->list();

        // Assert the result is as expected
        $this->assertIsArray($result);
        $this->assertCount(3, $result);
        $this->assertEquals('User 1', $result[1]['name']);
        $this->assertEquals('User 2', $result[2]['name']);
        $this->assertEquals('User 3', $result[3]['name']);

        // Assert the request was made with the correct method and path
        $this->assertRequestMade('GET', 'users');
    }

    /**
     * Test the get method
     */
    public function testGet(): void
    {
        // Mock the authentication response
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));

        // Mock the get response
        $mockData = [
            'value' => [
                ['id' => 1, 'name' => 'User 1']
            ]
        ];
        $this->addMockResponse(200, [], json_encode($mockData));

        // Set up static properties
        ApiClient::setAuthURL('https://example.com/oauth/token');
        ApiClient::setClientID('test-client-id');
        ApiClient::setClientSecret('test-client-secret');
        ApiClient::setGrantType('client_credentials');
        ApiClient::setScope('read write');

        // Get an instance
        $apiClient = ApiClient::getInstance('users');

        // Call the get method
        $result = $apiClient->get(1);

        // Assert the result is as expected
        $this->assertIsArray($result);
        $this->assertEquals(1, $result['id']);
        $this->assertEquals('User 1', $result['name']);

        // Assert the request was made with the correct method and path
        $this->assertRequestMade('GET', 'users/1');
    }

    /**
     * Test the post method with dry run mode enabled
     */
    public function testPostWithDryRunEnabled(): void
    {
        // Mock the authentication response
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));

        // Set up static properties
        ApiClient::setAuthURL('https://example.com/oauth/token');
        ApiClient::setClientID('test-client-id');
        ApiClient::setClientSecret('test-client-secret');
        ApiClient::setGrantType('client_credentials');
        ApiClient::setScope('read write');

        // Get an instance
        $apiClient = ApiClient::getInstance('users');

        // Ensure dry run mode is enabled
        $apiClient->setDry(true);

        // Call the post method
        $data = ['name' => 'New User'];
        $result = $apiClient->post($data);

        // Assert the result is null (no actual request was made)
        $this->assertNull($result);

        // Assert the status code is 200 (success)
        $this->assertEquals(200, $apiClient->getStatusCode());

        // Assert no POST request was actually made
        $this->assertCount(1, $this->requestHistory); // Only the authentication request
    }

    /**
     * Test the post method with dry run mode disabled
     */
    public function testPostWithDryRunDisabled(): void
    {
        // Mock the authentication response
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));

        // Mock the post response
        $mockData = ['id' => 4, 'name' => 'New User'];
        $this->addMockResponse(201, [], json_encode($mockData));

        // Set up static properties
        ApiClient::setAuthURL('https://example.com/oauth/token');
        ApiClient::setClientID('test-client-id');
        ApiClient::setClientSecret('test-client-secret');
        ApiClient::setGrantType('client_credentials');
        ApiClient::setScope('read write');

        // Get an instance
        $apiClient = ApiClient::getInstance('users');

        // Disable dry run mode
        $apiClient->setDry(false);

        // Call the post method
        $data = ['name' => 'New User'];
        $result = $apiClient->post($data);

        // Assert the result is a Response object
        $this->assertInstanceOf(Response::class, $result);

        // Assert the status code is 201 (created)
        $this->assertEquals(201, $apiClient->getStatusCode());

        // Assert the request was made with the correct method and path
        $this->assertRequestMade('POST', 'users');
    }

    /**
     * Test the pagination handling in getAllPages method
     */
    public function testPaginationHandling(): void
    {
        // Mock the authentication response
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));

        // Mock the first page response
        $page1Data = [
            'value' => [
                ['id' => 1, 'name' => 'User 1'],
                ['id' => 2, 'name' => 'User 2']
            ],
            '@odata.nextLink' => 'users?page=2'
        ];
        $this->addMockResponse(200, [], json_encode($page1Data));

        // Mock the second page response
        $page2Data = [
            'value' => [
                ['id' => 3, 'name' => 'User 3'],
                ['id' => 4, 'name' => 'User 4']
            ]
        ];
        $this->addMockResponse(200, [], json_encode($page2Data));

        // Set up static properties
        ApiClient::setAuthURL('https://example.com/oauth/token');
        ApiClient::setClientID('test-client-id');
        ApiClient::setClientSecret('test-client-secret');
        ApiClient::setGrantType('client_credentials');
        ApiClient::setScope('read write');

        // Get an instance
        $apiClient = ApiClient::getInstance('users');

        // Call the list method
        $result = $apiClient->list();

        // Assert the result is as expected
        $this->assertIsArray($result);
        $this->assertCount(4, $result);
        $this->assertEquals('User 1', $result[1]['name']);
        $this->assertEquals('User 2', $result[2]['name']);
        $this->assertEquals('User 3', $result[3]['name']);
        $this->assertEquals('User 4', $result[4]['name']);

        // Assert that two GET requests were made
        $this->assertCount(3, $this->requestHistory); // Auth + 2 pages
    }
    /**
     * Test the put method
     */
    public function testPut(): void
    {
        // Mock the authentication response
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));

        // Mock the put response
        $mockData = ['id' => 1, 'name' => 'Updated User'];
        $this->addMockResponse(200, [], json_encode($mockData));

        // Set up static properties
        ApiClient::setAuthURL('https://example.com/oauth/token');
        ApiClient::setClientID('test-client-id');
        ApiClient::setClientSecret('test-client-secret');
        ApiClient::setGrantType('client_credentials');
        ApiClient::setScope('read write');

        // Get an instance
        $apiClient = ApiClient::getInstance('users');

        // Disable dry run mode
        $apiClient->setDry(false);

        // Call the put method
        $data = ['name' => 'Updated User'];
        $result = $apiClient->put(1, $data);

        // Assert the result is a Response object
        $this->assertInstanceOf(Response::class, $result);

        // Assert the status code is 200 (success)
        $this->assertEquals(200, $apiClient->getStatusCode());

        // Assert the request was made with the correct method and path
        $this->assertRequestMade('PUT', 'users/1');
    }

    /**
     * Test the delete method
     */
    public function testDelete(): void
    {
        // Mock the authentication response
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));

        // Mock the delete response
        $this->addMockResponse(204, [], '');

        // Set up static properties
        ApiClient::setAuthURL('https://example.com/oauth/token');
        ApiClient::setClientID('test-client-id');
        ApiClient::setClientSecret('test-client-secret');
        ApiClient::setGrantType('client_credentials');
        ApiClient::setScope('read write');

        // Get an instance
        $apiClient = ApiClient::getInstance('users');

        // Disable dry run mode
        $apiClient->setDry(false);

        // Call the delete method
        $result = $apiClient->delete(1);

        // Assert the result is a Response object
        $this->assertInstanceOf(Response::class, $result);

        // Assert the status code is 204 (no content)
        $this->assertEquals(204, $apiClient->getStatusCode());

        // Assert the request was made with the correct method and path
        $this->assertRequestMade('DELETE', 'users/1');
    }

    /**
     * Test the patch method
     */
    public function testPatch(): void
    {
        // Mock the authentication response
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));

        // Mock the patch response
        $mockData = ['id' => 1, 'name' => 'Partially Updated User'];
        $this->addMockResponse(200, [], json_encode($mockData));

        // Set up static properties
        ApiClient::setAuthURL('https://example.com/oauth/token');
        ApiClient::setClientID('test-client-id');
        ApiClient::setClientSecret('test-client-secret');
        ApiClient::setGrantType('client_credentials');
        ApiClient::setScope('read write');

        // Get an instance
        $apiClient = ApiClient::getInstance('users');

        // Disable dry run mode
        $apiClient->setDry(false);

        // Call the patch method
        $data = ['name' => 'Partially Updated User'];
        $result = $apiClient->patch(1, $data);

        // Assert the result is a Response object
        $this->assertInstanceOf(Response::class, $result);

        // Assert the status code is 200 (success)
        $this->assertEquals(200, $apiClient->getStatusCode());

        // Assert the request was made with the correct method and path
        $this->assertRequestMade('PATCH', 'users/1');
    }

    /**
     * Test the create method
     */
    public function testCreate(): void
    {
        // Mock the authentication response
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));

        // Mock the create response
        $mockData = ['id' => 4, 'name' => 'Created User'];
        $this->addMockResponse(201, [], json_encode($mockData));

        // Set up static properties
        ApiClient::setAuthURL('https://example.com/oauth/token');
        ApiClient::setClientID('test-client-id');
        ApiClient::setClientSecret('test-client-secret');
        ApiClient::setGrantType('client_credentials');
        ApiClient::setScope('read write');

        // Get an instance
        $apiClient = ApiClient::getInstance('users');

        // Disable dry run mode
        $apiClient->setDry(false);

        // Call the create method
        $data = ['name' => 'Created User'];
        $result = $apiClient->create($data);

        // Assert the result is a Response object
        $this->assertInstanceOf(Response::class, $result);

        // Assert the status code is 201 (created)
        $this->assertEquals(201, $apiClient->getStatusCode());

        // Assert the request was made with the correct method and path
        $this->assertRequestMade('POST', 'users');
    }

    /**
     * Test error handling in the request method
     */
    public function testErrorHandling(): void
    {
        // Mock the authentication response
        $this->addMockResponse(200, [], json_encode(['access_token' => 'mock-token']));

        // Mock a 404 error response
        $this->addMockResponse(404, [], json_encode(['error' => 'Not Found']));

        // Set up static properties
        ApiClient::setAuthURL('https://example.com/oauth/token');
        ApiClient::setClientID('test-client-id');
        ApiClient::setClientSecret('test-client-secret');
        ApiClient::setGrantType('client_credentials');
        ApiClient::setScope('read write');

        // Get an instance
        $apiClient = ApiClient::getInstance('users');

        // Call the get method with a non-existent ID
        $result = $apiClient->get(999);

        // Assert the result is an empty array (as defined in the getAllPages method)
        $this->assertEmpty($result);

        // Assert the status code is 404
        $this->assertEquals(404, $apiClient->getStatusCode());
    }
}
