# AMG API Client

A flexible PHP API client library for making HTTP requests to various endpoints with OAuth2 authentication support.

## Overview

The AMG API Client is a PHP library that provides a robust and flexible implementation for interacting with RESTful APIs. It implements a multiton pattern to manage different API endpoint connections efficiently.

### Key Features

- OAuth2 authentication with token management
- Support for standard HTTP methods (GET, POST, PUT, PATCH, DELETE)
- Pagination handling for large result sets
- "Dry run" mode to prevent destructive operations during testing
- Logging of API operations

## Installation

You can install the package via composer:

```bash
composer require amggroup/apiclient
```

## Usage

### Basic Usage

```php
// Set authentication parameters
ApiClient::setAuthURL('https://api.example.com/oauth/token');
ApiClient::setClientID('your-client-id');
ApiClient::setClientSecret('your-client-secret');
ApiClient::setGrantType('client_credentials');
ApiClient::setScope('read write');

// Get an instance for a specific endpoint
$api = ApiClient::getInstance('users');

// Disable dry run mode if you want to perform write operations
$api->setDry(false);

// Perform API operations
$users = $api->list();
$user = $api->get(123);
$response = $api->post(['name' => 'New User']);
```

### Available Methods

- `list($reIndex = 'id')`: Get all records from the endpoint, optionally reindexed by a specific field
- `get($id)`: Get a single record by ID
- `post($data)`: Create a new record using POST
- `put($id, $data)`: Update a record using PUT
- `patch($id, $data)`: Partially update a record using PATCH
- `delete($id)`: Delete a record
- `create($data)`: Create a new record (semantic alias for post)

### Dry Run Mode

The API client includes a "dry run" mode that prevents destructive operations (POST, PUT, PATCH, DELETE) from being executed. This is useful for testing API interactions without making actual changes.

```php
// Enable dry run mode (enabled by default)
$api->setDry(true);

// Disable dry run mode to perform write operations
$api->setDry(false);
```

## Configuration

### Authentication Configuration

The API client supports OAuth2 authentication. You can set the authentication parameters using the following static methods:

```php
ApiClient::setAuthURL('https://api.example.com/oauth/token');
ApiClient::setClientID('your-client-id');
ApiClient::setClientSecret('your-client-secret');
ApiClient::setGrantType('client_credentials');
ApiClient::setScope('read write');
```

Alternatively, you can provide these parameters when getting an instance:

```php
$api = ApiClient::getInstance(
    'users',
    'your-client-id',
    'your-client-secret',
    'client_credentials',
    'read write',
    'https://api.example.com/oauth/token'
);
```

### API Authentication Field Names

By default, the API client uses standard field names for OAuth2 authentication:
- `client_id` for the client ID
- `client_secret` for the client secret
- `grant_type` for the grant type
- `scope` for the scope

However, some APIs may use different field names. You can configure these field names in your configuration file:

```yaml
# Example configuration
api:
  clientIDKey: client_id       # Field name for client ID
  clientSecretKey: client_secret  # Field name for client secret
  grantTypeKey: grant_type     # Field name for grant type
  scopeKey: scope              # Field name for scope
```

If these configuration values are not set, the API client will use the default field names.

### Authentication Field Values

The values for these fields depend on your API provider:

- **client_id**: The client identifier issued to the client during the registration process
- **client_secret**: The client secret issued to the client during the registration process
- **grant_type**: The OAuth2 grant type (e.g., 'client_credentials', 'password', 'authorization_code')
- **scope**: A space-delimited list of scopes that define the access permissions

Example values:
```php
ApiClient::setClientID('abc123');
ApiClient::setClientSecret('xyz789');
ApiClient::setGrantType('client_credentials');
ApiClient::setScope('read write');
```

## Dependencies

- amggroup/logger: For logging API operations
- amggroup/config: For configuration management
- guzzlehttp/guzzle: For HTTP requests
- psr/http-message: For HTTP message interfaces

## License

This project is licensed under the MIT License.