<?php

namespace AmgGroup\Tests;

use PHPUnit\Framework\TestCase as BaseTestCase;
use AmgGroup\Config;
use AmgGroup\Logger;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;
use GuzzleHttp\Middleware;
use Psr\Http\Message\RequestInterface;

/**
 * Base TestCase for ApiClient tests
 * 
 * This class provides common setup and utility methods for testing the ApiClient class.
 * It includes methods for mocking HTTP responses, Config, and Logger classes.
 */
class TestCase extends BaseTestCase
{
    /**
     * @var array Container for history of requests
     */
    protected $requestHistory = [];

    /**
     * @var MockHandler Mock handler for Guzzle client
     */
    protected $mockHandler;

    /**
     * @var Client Mocked Guzzle client
     */
    protected $mockClient;

    /**
     * Set up test environment
     */
    protected function setUp(): void
    {
        parent::setUp();

        // Reset request history
        $this->requestHistory = [];

        // Create a mock handler
        $this->mockHandler = new MockHandler();

        // Create a history middleware
        $history = Middleware::history($this->requestHistory);

        // Create a handler stack with the mock handler
        $handlerStack = HandlerStack::create($this->mockHandler);

        // Add the history middleware to the handler stack
        $handlerStack->push($history);

        // Create a client with the handler stack
        $this->mockClient = new Client(['handler' => $handlerStack]);

        // Mock Config class
        $this->mockConfig();

        // Mock Logger class
        $this->mockLogger();
    }

    /**
     * Mock the Config class
     */
    protected function mockConfig(): void
    {
        $configMock = $this->createMock(Config::class);
        $configMock->method('configIsLoaded')->willReturn(true);
        $configMock->method('get')->willReturnCallback(function ($key) {
            $configValues = [
                'graph.tenant_id' => 'mock-tenant-id',
                'api.clientIDKey' => 'client_id',
                'api.clientSecretKey' => 'client_secret',
                'api.grantTypeKey' => 'grant_type',
                'api.scopeKey' => 'scope'
            ];

            return $configValues[$key] ?? null;
        });

        // Create a mock for the Config class that will be returned by getInstance
        // This approach uses reflection to set a static property, which is a common way
        // to mock singleton/multiton patterns in PHP
        try {
            $configReflection = new \ReflectionClass(Config::class);
            if ($configReflection->hasProperty('instance')) {
                $instanceProperty = $configReflection->getProperty('instance');
                $instanceProperty->setAccessible(true);
                $instanceProperty->setValue(null, $configMock);
            }
        } catch (\ReflectionException $e) {
            // If we can't mock it this way, we'll need to use a different approach
            // For now, we'll just note that this might fail in actual tests
            $this->markTestSkipped('Unable to mock Config class: ' . $e->getMessage());
        }
    }

    /**
     * Mock the Logger class
     */
    protected function mockLogger(): void
    {
        $loggerMock = $this->createMock(Logger::class);

        // Create mock objects for info and error properties
        $infoLoggerMock = $this->createMock(\stdClass::class);
        $errorLoggerMock = $this->createMock(\stdClass::class);

        // Set up method mocks for the info and error objects
        $infoLoggerMock->method('apiClient')->willReturn(null);
        $infoLoggerMock->method('apiRequest')->willReturn(null);
        $errorLoggerMock->method('apiClient')->willReturn(null);
        $errorLoggerMock->method('apiRequest')->willReturn(null);

        // Set the properties on the logger mock
        $loggerMock->info = $infoLoggerMock;
        $loggerMock->error = $errorLoggerMock;

        // Create a mock for the Logger class that will be returned by getInstance
        // This approach uses reflection to set a static property, which is a common way
        // to mock singleton/multiton patterns in PHP
        try {
            $loggerReflection = new \ReflectionClass(Logger::class);
            if ($loggerReflection->hasProperty('instance')) {
                $instanceProperty = $loggerReflection->getProperty('instance');
                $instanceProperty->setAccessible(true);
                $instanceProperty->setValue(null, $loggerMock);
            }
        } catch (\ReflectionException $e) {
            // If we can't mock it this way, we'll need to use a different approach
            // For now, we'll just note that this might fail in actual tests
            $this->markTestSkipped('Unable to mock Logger class: ' . $e->getMessage());
        }
    }

    /**
     * Add a mock response to the queue
     *
     * @param int $status HTTP status code
     * @param array $headers Response headers
     * @param string $body Response body
     */
    protected function addMockResponse(int $status = 200, array $headers = [], string $body = '{}'): void
    {
        $this->mockHandler->append(new Response($status, $headers, $body));
    }

    /**
     * Get the last request from history
     *
     * @return RequestInterface|null
     */
    protected function getLastRequest(): ?RequestInterface
    {
        if (!empty($this->requestHistory)) {
            return end($this->requestHistory)['request'];
        }

        return null;
    }

    /**
     * Assert that a request was made with the given method and path
     *
     * @param string $method HTTP method
     * @param string $path Request path
     */
    protected function assertRequestMade(string $method, string $path): void
    {
        $found = false;

        foreach ($this->requestHistory as $transaction) {
            $request = $transaction['request'];
            if ($request->getMethod() === $method && strpos((string)$request->getUri(), $path) !== false) {
                $found = true;
                break;
            }
        }

        $this->assertTrue($found, "No request was made with method $method and path $path");
    }
}
